package plumstone;			// Commented out during development phase
import javax.sound.midi.spi.*;
import javax.sound.midi.*;
import com.apple.audio.util.*;
import com.apple.audio.midi.*;
import com.apple.audio.util.CAMemoryObject.*;

/**
  <p>Class StoneMidiDeviceProvider.java of project PlumStone
  
  <p>This class is the main server in the Core Midi implementation of javax.sound.midi
  Midi input and output classes.  It provides a list of Midi Transmitters and Receivers
  supported by the current Apple Core Audio Midi setup, plus methods to find suitable
  Transmitter and Receiver objects for Midi I/O.
  
  <p>Access to this class is via the Java server mechanism.  The programmer uses the
  standard MidiSystem to obtain a list of supported midi devices.  The required
  Transmitter/Receiver is then chosen from this list.  A particularly convenient way
  is to present the program user with the device list (in a combo box) and allow the
  user to choose.
  
  <p>An unfortunate side effect of the server invocation method is that the constructor
  for this class may be called several times.  For this reason, many of the attributes are
  marked static so that they retain their values across these multiple invocations.
  
  <p>Version 1.2: No major mods except for version string.
 
  @author Bob Lang
  @version Fri Feb 4 2004 version 1.2
*/
public class StoneMidiDeviceProvider extends MidiDeviceProvider {

  /**
    Standard version string for this project  
  */
  public static final String
    version = "Version 1.2";
  
  // Count the number of invocations of the constructor
  private static int invocations = 0;

  // **** javax.sound.midi data ****

  // Arrays for the supported devices and their associated information structures
  private static MidiDevice [] deviceArray;
  private static MidiDevice.Info [] infoArray;
  private static int deviceCount = 0;

  // **** Apple Core Audio Midi data ****

  // Access to the Core Audio setup
  private static MIDISetup coreAudioSetup;

  // Counters for Core Audio sources and destinations
  private static int numberOfSources;
  private static int numberOfDestinations;

  /**
    Constructor this class which identifies all the connected Midi devices, draws up
    a suitable list and then creates a (javax.sound.midi) Transmitter or Receiver
    for each one.
    
    <p>In normal operation, this constructor isn't called directly by the user - instead
    it is activated via the Java server system.  An unfortunate side effect is that this
    constructor may be called several times.  For this reason, the method only does the
    bulk of its work on its first invocation and all the attributes are marked static.
  */
  public StoneMidiDeviceProvider () {
    super ();

    // Count the number of invocations of this constructor
    invocations++;

    // Create devices only on first invocation
    if (invocations == 1) {
      try {
        // Get the Core Audio set and the number of Midi sources and destinations
        coreAudioSetup = MIDISetup.getCurrent ();
        numberOfSources = coreAudioSetup.getNumberOfSources ();
        numberOfDestinations = coreAudioSetup.getNumberOfDestinations ();
        deviceCount = numberOfSources + numberOfDestinations;

        // Create arrays exactly to size
        deviceArray = new MidiDevice [deviceCount];
        infoArray = new MidiDevice.Info [deviceCount];
        int deviceNumber = 0;

        // Identify every Core Audio source and make into javax.sound.midi Transmitter
        for (int i = 0; i < numberOfSources; i++) {
          // Create a transmitter and get corresponding device information
          deviceArray [deviceNumber] = new StoneMidiTransmitter (i);
          infoArray[deviceNumber] = deviceArray [deviceNumber].getDeviceInfo ();
          deviceNumber++;
        } // for

        // Identify every Core Audio destination and make into javax.sound.midi Receiver
        for (int i = 0; i < numberOfDestinations; i++) {
          // Create a transmitter and get corresponding device information
          deviceArray [deviceNumber] = new StoneMidiReceiver (i);
          infoArray[deviceNumber] = deviceArray [deviceNumber].getDeviceInfo ();
          deviceNumber++;
        } // for
      }
      catch (Exception e) {
        System.out.println ("StoneMidiDeviceProvider () reports exception:");
        System.out.println (e);
      }
    } // if first invocation
  } // StoneMidiDeviceProvider ()

  /**
    Returns the appropriate MidiDevice corresponding to the given Info.
    
    <p>The parameter should be one of the entries returned by getDeviceInfo rather than
    an arbitrary structure made up by the programmer
  */
  public MidiDevice getDevice (MidiDevice.Info device) {
    // Search all information for entry matching the device
    for (int i=0; i< deviceCount; i++) {
      if (infoArray [i].equals (device)) {
        return deviceArray [i];
      } // if
    } // for

    // The device must be illegal, or unknown to this provider
    throw
      new IllegalArgumentException ("PlumStone - Unsupported device" + device.toString ());
  } // getDevice ()

  /**
    Returns a complete list of supported devices
  */
  public MidiDevice.Info [] getDeviceInfo () {
    return infoArray;
  } // getDeviceInfo ()
} // StoneMidiDeviceProvider





  

