package plumstone;			// Commented out during development phase
import com.apple.audio.midi.*;
import com.apple.audio.util.*;
import javax.sound.midi.*;

/**
  <p>Class StoneMidiMessageBuilder.java of project PlumStone
  
  <p>An object of this class collect midi bytes, formulates them into javax.sound.midi
  MidiMessages (using appropriate sub-classes) and then passes the completed messages
  to the registered javax.sound.midi Receiver object.
  
  <p>Additionally, the class contains some useful static methods.
 
  <p>Version 1.2: Added short message correction to cope with situations where the Short Message length
  is not consistent with the message contents
  
  @author Bob Lang
  @version Fri Feb 4 2005 version 1.2
*/
public class StoneMidiMessageBuilder {

  // Channel messages (channel number in lower four bits)
  public static final int
    NOTE_OFF = ShortMessage.NOTE_OFF, 													// 0x80 or 128
    NOTE_ON = ShortMessage.NOTE_ON,															// 0x90 or 144
    CONTROL_CHANGE = ShortMessage.CONTROL_CHANGE, 							// 0xB0 or 176
    POLY_PRESSURE = ShortMessage.POLY_PRESSURE,									// 0xA0 or 128
    PROGRAM_CHANGE = ShortMessage.PROGRAM_CHANGE,								// 0xC0 or 192
    CHANNEL_PRESSURE = ShortMessage.CHANNEL_PRESSURE, 					// 0xD0 or 208
    PITCH_BEND = ShortMessage.PITCH_BEND,												// 0xE0 or 224
    SYSTEM_COMMON = 0xF0;

  // Non-channel messages (top four bits = F)
  public static final int
    SYSTEM_EXCLUSIVE = SysexMessage.SYSTEM_EXCLUSIVE,						// 0xF0 or 240
    MIDI_TIME_CODE = ShortMessage.MIDI_TIME_CODE,								// 0xF1 or 241
    SONG_POSITION_POINTER = ShortMessage.SONG_POSITION_POINTER,	// 0xF2 or 242
    SONG_SELECT = ShortMessage.SONG_SELECT, 										// 0xF3 or 243
    TUNE_REQUEST = ShortMessage.TUNE_REQUEST,										// 0xF6 or 246
    END_OF_EXCLUSIVE = ShortMessage.END_OF_EXCLUSIVE, 					// 0xF7 or 247
    CONTINUE_EXCLUSIVE = END_OF_EXCLUSIVE,											// Also 247
    TIMING_CLOCK = ShortMessage.TIMING_CLOCK,										// 0xF8 or 248
    START = ShortMessage.START, 																// 0xFA or 250
    CONTINUE = ShortMessage.CONTINUE,														// 0xFB or 251
    STOP= ShortMessage.STOP,																		// 0xFC or 252
    ACTIVE_SENSING = ShortMessage.ACTIVE_SENSING, 							// 0xFE or 254
    SYSTEM_RESET = ShortMessage.SYSTEM_RESET,										// 0xFF or 255
    META = MetaMessage.META;																		// Also 255

  // Define the receiver object for the messages
  private Receiver receiver;

  // Status codes during the building of a message
  private static final int
    BLD_NO_MESSAGE = 0,					// Not building any message
    BLD_SHORT_MESSAGE = 1,			// Now building a short message
    BLD_SYSEX_MESSAGE = 2;			// Now building a system exclusive message

  // Buffer for all messages
  private static final int MAX_BUFFER_LENGTH = 1024;
  private byte [] buffer = new byte [MAX_BUFFER_LENGTH];
  private int bufferLength = 0;
  private int targetLength = 0;
  private int buildStatus;

  // Global time stamp
  private long timestamp;

  /**
    Constructor for the class.
  */
  public StoneMidiMessageBuilder () {
    resetBuffer ();
    buildStatus = BLD_NO_MESSAGE;
  } // StoneMidiMessageBuilder ()

  /**
    <p>Method which accepts a Core Audio Midi Packet, breaks it down into individual data
    bytes and rebuilds them into javax.sound.midi messages.  A complete message may take
    several midi packets, or there may be several messages in one packet.  As each message
    is built up, it is passed on to the registered Receiver.
    
    <p>As each packet is received, its time stamp is stored.  When a message is complete,
    then it is forwarded using the time stamp of the midi packet in which it was completed.

    <p>In principle, the user can change the registered receiver at any time (although
    this is is not recommended), hence the receiver object is sent with each midi packet.
    When a message is complete, it is forwarded to the current receiver.
  */
  public void sendMessagePacket (MIDIPacket packet,
                                 Receiver theReceiver,
                                 long theTimestamp)
  {
    // Save the latest receiver and time stamp
    receiver = theReceiver;
    timestamp = theTimestamp;
    
    // Extract data length and raw data from the packet
    int length = packet.getLength ();
    MIDIData packetData = packet.getData ();

    // Process each byte of packet data
    for (int i=0; i<length; i++) {
      // Get the current byte as an integer
      int data = packetData.getByteAt (i) & 0xFF;

      // Is this a data byte
      if (data < 0x80) {
        // Store data in the byte buffer
        updateBuffer (data);

        // All the required bytes found?
        if (bufferLength == targetLength ) {
          // Assume it's a short message
          sendShortMessage ();
        } // if
      } // if data byte received

      // Is this a system exclusive message?
      else if (data == SYSTEM_EXCLUSIVE) {
        // Remember what we're doing
        buildStatus = BLD_SYSEX_MESSAGE;
        resetBuffer ();
        targetLength = -1;
        
        // Store the byte in the buffer
        updateBuffer (data);
      }

      // Is this the end of a system exclusive message
      else if (data == END_OF_EXCLUSIVE) {
        // Store the data and send the whole message
        updateBuffer (data);
        sendSysexMessage ();
      }

      // Is this a channel message status byte?
      else if (data < 0xF0) {
        // Start a new message no matter what the context
        resetBuffer ();
        targetLength = getNominalMessageLength (data);
        buildStatus = BLD_SHORT_MESSAGE;

        // Store the byte in the buffer
        updateBuffer (data);

        // Some messages are only one byte long
        if (targetLength == 1) {
          sendShortMessage ();
        } // if
      }

      // It's not data and it's not a channel message, so it must be a system message
      else {
        // Throw this byte away if we're not ready for a new message
        if (buildStatus == BLD_NO_MESSAGE || buildStatus == BLD_SHORT_MESSAGE) {
          // Ready the buffer for a new message
          resetBuffer ();
          targetLength = getNominalMessageLength (data);
          buildStatus = BLD_SHORT_MESSAGE;

          // Store the byte in the buffer
          updateBuffer (data);

          // Some messages are only one byte long
          if (targetLength == 1) {
            sendShortMessage ();
          } // if
        } // if
      } // else
    } // for
  } // sendMessagePacket ()

  /**
    Service method to create an equivalent java.sound.midi.MidiMessage from
   a Core Audio MIDIPacket
   */
  public static MIDIPacketList makeMidiPacketList (MidiMessage message) {
    MIDIPacketList packetList = null;

    // Find the length and get the data bytes
    int length = message.getLength ();
    byte [] rawDataBytes = message.getMessage ();

    // Create an empty Core Audio Midi Data object
    MIDIData midiData = MIDIData.newMIDIRawData (length);
    midiData.addRawData (rawDataBytes);

    // Embed the message into a Core Audio Midi Packet List
    try {
      packetList = new MIDIPacketList (0L, midiData);
    }
    catch (Exception e) {
      System.out.println ("Exception formulating MIDIPacketList");
      System.out.println (e);
    }
    // return the final result
    return packetList;
  } // makeMidiPacketList
  
  /**
     Reset the buffer to accept a new message
  */
  private void resetBuffer () {
    bufferLength = 0;
    targetLength = 0;
  } // resetBuffer ()

  /**
    Partially reset the buffer so that running status is retained for the next message.
  */
  private void resetForRunningStatus () {
    // Retain existing status byte at position [0]
    bufferLength = 1;
  } // resetForRunningStatus ()

  /**
    Add a new byte into the buffer.  If the buffer's maximum length is exceeded then the
    whole of the current message is discarded
  */
  private void updateBuffer (int data) {
    // Check that there's still room in the buffer
    if (bufferLength < MAX_BUFFER_LENGTH) {
      // Add the data to the end of the buffer
      buffer [bufferLength] = (byte) (data & 0xFF);
      bufferLength ++;
    } // if
    else {
      // Something is seriously wrong here - probably a sysex has been started
      // but it's either too long, or the ending byte 0F7 has been lost

      // Are we building a Sysex message?
      if (buildStatus == BLD_SYSEX_MESSAGE) {
        // Output what we have so far, then reset for a contination
        sendSysexMessage ();
        buildStatus = BLD_SYSEX_MESSAGE;
        resetBuffer ();
        targetLength = -1;
        
        // Put a new starting byte and the current data in the buffer
        updateBuffer (CONTINUE_EXCLUSIVE);
        updateBuffer (data);
      }
      else {
        // The only reasonable action is to discard the buffer and reset for new input
        resetBuffer ();
        buildStatus = BLD_NO_MESSAGE;
      }
    } // else
  } // updateBuffer ()

  /**
    Use the contents of the buffer to create a ShortMessage and send it to the
    registered Receiver.  After the message has been sent, the buffer is reset
    either fully, or to allow the next message to use running status.
  */
  private void sendShortMessage () {
    try {
      // Check that the message really is three bytes or less
      if (bufferLength <= 3) {
        int status = buffer [0] & 0xFF;
        ShortMessage message = new ShortMessage ();
        switch (bufferLength) {
          case 1:
            message.setMessage (status);
            break;
          case 2:
            message.setMessage (status, (int) buffer [1], 0);
            break;
          case 3:
            message.setMessage (status, (int) buffer [1], (int) buffer [2]);
            break;
          default:
            message.setMessage (status);
        } // switch
  
        // Send the message to the receiver
        receiver.send (message, timestamp);
      } // if
    } // try
    catch (Exception e) {
      System.out.println ("Exception building Short Midi Message");
      System.out.println (e);
    }  // catch

    // Channel messages may have running status
    if ((buffer [0] & 0xF0) != 0xF0) {
      // Allow for possibility of running status
      resetForRunningStatus ();
      buildStatus = BLD_SHORT_MESSAGE;
    } // if channel message
    else {
      // Reset the buffer and message build status
      resetBuffer ();
      buildStatus = BLD_NO_MESSAGE;
    } // non-channel message
  } // sendShortMessage ()

  /**
    Use the contents of the buffer to create a system exclusive message and send it to the
    registered Receiver.  After the message has been sent, the buffer is fully reset.
  */
  private void sendSysexMessage () {
    try {
      // Create the system exclusive message
      SysexMessage message = new SysexMessage ();
      message.setMessage (buffer, bufferLength);
  
      // Send the message to the receiver
      receiver.send (message, timestamp);
    } // try
    catch (Exception e) {
      System.out.println ("Exception building System Exclusive Message");
      System.out.println (e);
    }  // catch

    // Reset the buffer and message build status
    resetBuffer ();
    buildStatus = BLD_NO_MESSAGE;
  } // sendSysexMessage ()
  
  /**
    Performs any necessary correction of message length and returns the corrected message.  If no corrections
    are required or possible then the original message is returned unchanged.    
  */
  public static MidiMessage shortMessageCorrection (MidiMessage inMessage) {
    // Assume we won't need any error correction
    MidiMessage msg = inMessage;
    
    // Can only work with short messages
    if (inMessage instanceof ShortMessage) {
      // Cast to appropriate short message type
      ShortMessage sm = (ShortMessage) inMessage;
      ShortMessage outSm = new ShortMessage ();
      
      // Get the status and the message length
      int status = sm.getCommand () + sm.getChannel ();
      int actualLength = inMessage.getLength ();
      int nominalLength = getNominalMessageLength (status);

      // Check that the actual and nominal lengths are in agreement
      if (actualLength != nominalLength) {
        try {
          // Build up the corrected message
          if (nominalLength == 1) {
            outSm.setMessage (status);
            msg = outSm;
          }
          else if (nominalLength == 2) {
            outSm.setMessage (status, sm.getData1 (), 0);
            msg = outSm;
          }
          else if (nominalLength == 3) {
            outSm.setMessage (status, sm.getData1 (), sm.getData2 ());
            msg = outSm;
          } 
          else {
            // Some strange message - can't fix it so just hand it back
            msg = inMessage;
          }
        }
        catch (InvalidMidiDataException e) {
          // Just go back to the original message
          msg = inMessage;
        }
      } // Lengths don't agree
    } // if ShortMessage
    return msg;
  } // shortMessageCorrection ()

  /**
    Returns the *nominal* number of bytes in a message for the specified status byte.  For
    example, a NOTE_ON message has three bytes,
    
    <p>Sysex and meta messages do not have a fixed length and so -1 is returned.  When the
    supplied value is not a valid status byte, 0 is returned.
  */
  public static int getNominalMessageLength (int status) {
    // Split the supplied value into high and low nibbles
    int highNibble = (status & 0xF0);
    int lowNibble = status & 0x0F;

    // Identify channel messages
    switch (highNibble) {
      case NOTE_OFF:
        return 3;

      case NOTE_ON:
        return 3;

      case CONTROL_CHANGE:
        return 3;

      case POLY_PRESSURE:
        return 3;

      case PROGRAM_CHANGE:
        return 2;

      case CHANNEL_PRESSURE:
        return 2;

      case PITCH_BEND:
        return 3;

      default:
    } // switch

    // If we get here, then it may be a system common or real time message
    if (highNibble == SYSTEM_COMMON) {
      switch (status) {
        case SYSTEM_EXCLUSIVE:
          return -1;

        case MIDI_TIME_CODE:
          return 2;

        case SONG_POSITION_POINTER:
          return 3;

        case SONG_SELECT:
          return 2;

        case TUNE_REQUEST:
          return 1;

        case END_OF_EXCLUSIVE:
          return 0;

        case TIMING_CLOCK:
          return 1;

        case START:
          return 1;

        case CONTINUE:
          return 1;

        case STOP:
          return 1;

        case ACTIVE_SENSING:
          return 1;

        case SYSTEM_RESET:
          return 1;

        default:
      } // switch
    } // if

    // If we get here, then we've been called with a non-status byte!
    return 0;    
  } // getNominalMessageLength ()

} // StoneMidiMessageBuilder
