package plumstone;			// Commented out during development phase
import com.apple.audio.util.*;
import com.apple.audio.midi.*;
import com.apple.audio.util.CAMemoryObject.*;
import javax.sound.midi.*;

/**
  <p>Class StoneMidiReceiver.java of project PlumStone
  
  <p>This class implements a javax.sound.midi.Receiver (and MidiDevice) using the
  facilities provided by Apple's Core Audio Midi.
  
  <p>An object of this class sends Midi message to a physical MIDI interface.  The standard
  javax.sound.midi system includes time-stamping of midi messages, but it also allows
  devices to ignore time stamping and to send output as soon as it is received.  This
  receiver therefore sends any midi messages as soon as they are received, hence it is up 
  to the driving program to ensure correct timing.
  
  <p>On first opening, the object creates a Core Midi client and destination.
  Closing the object does not dispose of this client so re-opening after a close obtains the
  original client and device.  This uses more system resources but memory is cheap today.
  
  <p>Access to this class is via the Java MIDI service structure rather than by direct
  invocation.
 
  <p>Version 1.2: Attempts to correct ShortMessage errors where the message length does not 
  agree with the data within the message.
  
  @author Bob Lang
  @version Fri 4-Feb-2005 version 1.2
*/
public class StoneMidiReceiver implements MidiDevice, Receiver {

  // Version string
  private static final String
    version = StoneMidiDeviceProvider.version;

  // **** javax.sound.midi data ****

  // Information structure for this receiver
  private MidiDevice.Info info;

  // **** Apple Core Audio data ****

  // Core Audio client and Midi destination and output port
  private MIDIClient client;
  private MIDIEndpoint destination;
  private MIDIOutputPort outputPort;

  // A unique number for this receiver
  private int deviceNumber;

  // The midi device/transmitter has been opened for use
  private boolean initialised;
  private boolean deviceOpen;

  // Time when opened - used to provide time stamps
  private long openTime;

  // Descriptive strings
  private String deviceName, manufacturer;

  /**
    Constructor for this class.  Creates a StoneMidiReceiver which transmits midi data
    to an external midi hardware device.
    
    <p>Before it can be used, the open () method must be called.
    
    <p>The index number identifies the destination to be used in the current core audio
    setup.
  */
  public StoneMidiReceiver (int index) {
    super ();
    try {
      // Initialisation
      deviceOpen = false;
      initialised = false;
      deviceNumber = index;

      // Get the current Core Midi setup and the destination device from the index number
      MIDISetup setup = MIDISetup.getCurrent ();
      destination = MIDISetup.getDestination (index);

      // Build up the information strings
      deviceName =
        getStringProperty (destination,
                           MIDIConstants.kMIDIPropertyName,
                           "Unknown Output Device");
      deviceName += " (MidiOUT:" + index + ")";

      manufacturer =
        getStringProperty (destination,
                           MIDIConstants.kMIDIPropertyManufacturer,
                           "Unknown Manufacturer");

      // Create the information object for this transmitter
      Info localInfo = new Info (deviceName,
                                 manufacturer,
                                 "Midi OUT port:" + index,
                                 version);
      info = (MidiDevice.Info) localInfo;
    }
    catch (Exception e) {
      System.out.println ("StoneMidiReceiver () reports exception:");
      System.out.println (e);
    }
  } // StoneMidiReceiver ()

  /**
    Display device information in a useful form
   */
  public String toString () {
    String s = manufacturer + " " + deviceName;
    return s;
  } // toString ()

  // **** Methods required for MidiDevice interface ****

  /**
    Get information about the device
  */
  public MidiDevice.Info getDeviceInfo () {
    return info;
  } // getDeviceInfo ()

  /**
    Get the maximum number of receivers associated with this object
   */
  public int getMaxReceivers () {
    return 1;
  } // getMaxReceivers ()

  /**
    Get the maximum number of transmitters associated with this object
   */
  public int getMaxTransmitters () {
    return 0;
  } // getMaxTransmitters
  
  /**
    Get the current timestamp of the device in microseconds.  Note that time stamps
    are ignored on output
  */
  public long getMicrosecondPosition () {
    return 1000L * System.currentTimeMillis () - openTime;
  } // getMicrosecondPosition ()

  /**
    This device has no associated Transmitters and hence always throws an exception
  */
  public Transmitter getTransmitter () {
    throw
      new IllegalArgumentException ("Transmitter never available from " + info.toString ());
  } // getTransmitter ()

  /**
    Reports whether the device is open
   */
  public boolean isOpen () {
    return deviceOpen;
  } // isOpen ()

  /**
    Open the device and acquire any system resources necessary to may it operational
  */
  public void open () {
    try {
      // Has the device been opened previously?
      if (!initialised) {
        // Create the Core Midi client and the output port
        client = new MIDIClient (new CAFString ("PlumStone Client"),
                                 new StoneMidiClientCallBack ());
        // (outputPortCreate does not have a call back method)
        outputPort = client.outputPortCreate (new CAFString ("MidiOut" + deviceNumber));
      } // if not initialised

      // Record the time for time stamping and mark the device as now initialised and open
      openTime = 1000L * System.currentTimeMillis ();
      initialised = true;
      deviceOpen = true;
    }
    catch (Exception e) {
      System.out.print ("StoneMidiReceiver.open () reports exception");
      System.out.println (" associated with device " + info);
      System.out.println (e);
    }
  } // open ()

  /**
     Obtain a MIDI In receiver through which the MIDI device may receive MIDI data
  */
  public Receiver getReceiver () {
    return this;
  } // getReceiver ()

  // **** Methods required for Receiver interface ****

  /**
    Mark the device as closed, but don't actually return any system resources
  */
  public void close () {
    deviceOpen = false;
  } // close ()

  /**
     Send the midi message to the output device, correcting (if possible) malformed ShortMessages.
  */
  public void send (MidiMessage inMessage, long timeStamp) {
    try {
      // Ensure that the device is actually open
      if (deviceOpen) {
        // Make any message correction to bad ShortMessage objects
        MidiMessage message = StoneMidiMessageBuilder.shortMessageCorrection (inMessage);
        
        // Convert the midi message into Core Audio packet list
        MIDIPacketList packetList = StoneMidiMessageBuilder.makeMidiPacketList (message);
        outputPort.send (destination, packetList);
      } // if open
    }
    catch (Exception e) {
      System.out.print ("StoneMidiReceiver.send () reports exception");
      System.out.println (" associated with device " + info);
      System.out.println (e);
    }
  } // send ()

  /**
    Get a Core Audio device property string, if it exists.  If the property
   does not exist then the default string is returned.

   <p>This method is required as it simplifies processing of those devices
   which do not have a complete set of string properties specified by the
   manufacturer or implementor.
   */
  private String getStringProperty (MIDIEndpoint theDevice,
                                    CAFString theProperty,
                                    String defaultString)
  {
    // Create the returned result
    String result = defaultString;

    try {
      // Get the actual property string - if it exists
      result = theDevice.getStringProperty (theProperty).toString ();
    }
    catch (Exception e) {
      // The property doesn't exist so use the default
      result = defaultString;
    }

    // Return the final result
    return result;
  } // getStringProperty ()

  /**
    Nested Info class provides a link to the standard MidiDevice.Info as
    defined in javax.sound.midi
  */
  public static class Info extends MidiDevice.Info {
    protected Info (String s1, String s2, String s3, String s4) {
      super (s1, s2, s3, s4);
    } // Info ()
  } // Info class
} // StoneMidiReceiver

