package plumstone;			// Commented out during development phase
import com.apple.audio.util.*;
import com.apple.audio.midi.*;
import com.apple.audio.util.CAMemoryObject.*;
import javax.sound.midi.*;

/**
  <p>Class StoneMidiTransmitter.java of project PlumStone
  
  <p>This class implements a javax.sound.midi.Transmitter (and MidiDevice) using the
  facilities provided by Apple's Core Audio Midi.
  
  <p>An object of this class receives Midi packets from a physical interface, formats
  them into midi data messages and then passes them on to a registered Receiver.
  
  <p>On first opening, the object creates a Core Midi client and connects the midi source.
  Closing the object does not dispose of this client so re-opening after a close obtains the
  original client and device.  This uses more system resources but memory is cheap today.
  
  <p>Access to this class is via the Java MIDI service structure rather than by direct
  invocation.
  
  <p>Version 1.2: Corrected getMaxReceivers so that it now always returns 0.

  @author Bob Lang
  @version Fri March 18 2005 - version 1.2
*/
public class StoneMidiTransmitter implements MidiDevice, Transmitter, MIDIReadProc {

  // Version string
  private static final String
    version = StoneMidiDeviceProvider.version;

  // **** javax.sound.midi data ****

  // Information structure for this transmitter
  private MidiDevice.Info info;

  // Connected receiver
  private Receiver theReceiver;

  // Midi message builder
  StoneMidiMessageBuilder messageBuilder;

  // **** Apple Core Audio data ****

  // Core Audio client and Midi source and input port
  private MIDIClient client;
  private MIDIEndpoint source;
  private MIDIInputPort inputPort;

  // A unique number for this transmitter
  private int deviceNumber;

  // The midi device/transmitter has been opened for use
  private boolean initialised;
  private boolean deviceOpen;

  // Time when opened - used to provide time stamps
  private long openTime;

  // Descriptive strings
  private String deviceName, manufacturer;
  
  /**
    Constructor for this class.  Creates a StoneMidiTransmitter which receives midi data
    from an external midi hardware device.

    <p>Before it can be used, the open () method must be called.
    
    <p>The index number identifies the source to be used in the current core audio setup.
  */
  public StoneMidiTransmitter (int index) {
    super ();
    try {
      // Initialisation
      theReceiver = null;
      deviceOpen = false;
      initialised = false;
      deviceNumber = index;

      // Get the current Core Midi setup and the source device from the index number
      MIDISetup setup = MIDISetup.getCurrent ();
      source = MIDISetup.getSource (index);

      // Build up the information strings
      deviceName =
        getStringProperty (source,
                           MIDIConstants.kMIDIPropertyName,
                           "Unknown input device");
      deviceName += " (MidiIN:" + index + ")";
      manufacturer =
        getStringProperty (source,
                           MIDIConstants.kMIDIPropertyManufacturer,
                           "Unknown Manufacturer");

      // Create the information object for this transmitter
      Info localInfo = new Info (deviceName,
                                 manufacturer,
                                 "Midi IN port:" + index,
                                 version);
      info = (MidiDevice.Info) localInfo;
    }
    catch (Exception e) {
      System.out.println ("StoneMidiTransmitter () reports exception:");
      System.out.println (e);
    }
  } // StoneMidiTransmitter ()

  /**
    Display device information in a useful form
  */
  public String toString () {
    String s = manufacturer + " " + deviceName;
    return s;
  } // toString ()

  // **** Methods required for MidiDevice interface ****

  /**
    Get information about the device
  */
  public MidiDevice.Info getDeviceInfo () {
    return info;
  } // getDeviceInfo ()

  /**
    Get the maximum number of Midi in connections available on device
  */
  public int getMaxReceivers () {
    return 0;
  } // getMaxReceivers ()

  /**
    Get the maximum number of Midi out connections available on device
  */
  public int getMaxTransmitters () {
    return 1;
  }
  /**
    Get the current timestamp of the device in microseconds
  */
  public long getMicrosecondPosition () {
    return 1000L * System.currentTimeMillis () - openTime;
  } // getMicrosecondPosition ()

  /**
    Get a Midi Out connection from which the Midi device will transmit Midi data.
    In principle, the returned transmitter must be closed when the application has finished
    using it, but in practice this isn't necessary.
  */
  public Transmitter getTransmitter () {
    return this;
  } // getTransmitter ()

  /**
    Reports whether the device is open
  */
  public boolean isOpen () {
    return deviceOpen;
  } // isOpen
  
  /**
    Open the device and set up the Core Audio Midi system to make it operational
  */
  public void open () {
    try {
      // Has the device been opened previously?
      if (!initialised) {
        // Create the Core Midi client and the input port
        client = new MIDIClient (new CAFString ("PlumStone Client"),
                                 new StoneMidiClientCallBack ());
        inputPort = client.inputPortCreate (new CAFString ("MidiIn" + deviceNumber), this);
        inputPort.connectSource (source);

        // Create a midi message builder for this transmitter
        messageBuilder =  new StoneMidiMessageBuilder ();
      } // if not initialised

      // Record the time for time stamping and mark the device as now initialised and open
      openTime = 1000L * System.currentTimeMillis ();
      initialised = true;
      deviceOpen = true;
    }
    catch (Exception e) {
      System.out.print ("StoneMidiTransmitter.open () reports exception");
      System.out.println (" associated with device " + info);
      System.out.println (e);
    }
  } // open ()

  // **** Methods required for Transmitter interface ****

  /**
    Closes the device and returns any system resources it uses.  In practice, this isn't
    necessary.
  */
  public void close () {
    deviceOpen = false;
  } // close ()

  /**
    Get the current receiver to which this transmitter is delivering Midi messages
  */
  public Receiver getReceiver () {
    return theReceiver;
  } // getReceiver ()

  /**
    Set the receiver to which this transmitter will deliver Midi messages
  */
  public void setReceiver (Receiver receiver) {
    theReceiver = receiver;
  } // setReceiver ()

  // **** Method required for Core Audio MIDIReadProc interface ****

  /**
    Core Audio calls this method each time one or more midi messages are received from the
    external device.

    <p>If the device is open, then each midi message is converted from the Core Audio format
    to the corresponding javax.sound.midi format, and then passed on the registered Receiver
    object.
  */
  public void execute (MIDIInputPort port,
                       MIDIEndpoint srcEndPoint,
                       MIDIPacketList list)
  {
    // No action if the device isn't actually open
    if (deviceOpen) {
      // Caculate the timestamp for this message
      long timeStamp = 1000L * System.currentTimeMillis () - openTime;

      // Get the number of midi message packets and process each one
      int packetCount = list.numPackets ();
      for (int index=0; index<packetCount; index++) {
        // No point in further processing if there's no receiver
        if (theReceiver != null) {
          // Extract the next midi packet and pass on to the message builder
          MIDIPacket packet = list.getPacket (index);
          messageBuilder.sendMessagePacket (packet, theReceiver, timeStamp);
        } // if
      } // for
    } // if open
  } // execute ()

  /**
    Get a Core Audio device property string, if it exists.  If the property
    does not exist then the default string is returned.

    <p>This method is required as it simplifies processing of those devices
    which do not have a complete set of string properties specified by the
    manufacturer or implementor.
  */
  private String getStringProperty (MIDIEndpoint theDevice,
                                    CAFString theProperty,
                                    String defaultString)
  {
    // Create the returned result
    String result = defaultString;
    
    try {
      // Get the actual property string - if it exists
      result = theDevice.getStringProperty (theProperty).toString ();
    }
    catch (Exception e) {
      // The property doesn't exist so use the default
      result = defaultString;
    }

    // Return the final result
    return result;
  } // getStringProperty ()
  
  /**
    Nested Info class provides a link to the standard MidiDevice.Info as
    defined in javax.sound.midi
  */
  public static class Info extends MidiDevice.Info {
    protected Info (String s1, String s2, String s3, String s4) {
      super (s1, s2, s3, s4);
    } // Info ()
  } // Info class
} // StoneMidiTransmitter
